function [TPHAT] = SFlabFindPulseTimes(recBuffer,recChanList,playChanList,...
    sampRate,pulseFreq,pulseWidth,npt,time,plotOn)
%
% Function that finds the pulse times in a recording buffer.
% 
% function [TPHAT] = SFlabFindPulseTimes(recBuffer,recChanList,playChanList,...
%     sampRate,pulseFreq,pulseWidth,npt,time,plotOn)
%
% INPUTS:
% recBuffer    -  NxM-matrix with sound data, where M=length(mics) and
%                 N = (sample freq)*(time). Each column contains data from
%                 one of the microphones.
% recChanList  -  Vector with the recording channels that were used.
% playChanList -  Vector with the output channels that were used.
% sampRate     -  Sampling frequency.
% pulseFreq    -  Pulse frequencies.
% pulseWidth   -  Width of the pulses.
% npt          -  Vector with number of pulses per second for each output
%                 channel.
% time         -  Length in seconds of the recording.
% plotOn       -  Plot option, set 1 to plot, 0 otherwise.
%
% OUTPUTS:
% TPHAT        -  Matrix with pulse times, element (:,m,n) contains the
%                 pulse times for microphone m from speaker n.
%
%

% Martin Skoglund and Karl Granstrm
% 2009-03-24

% Number of microphones used
M = length(recChanList);

% Number of sound sources used
N = length(playChanList);
% Assuming there is one external sound source if no internal
if N == 0, N=1; playChanList = 1; end

% Maximum number of pulses
npmax = floor(max(npt*time));

% Allocate memory
TPHAT = zeros(npmax,M,N);

display('...finding pulse times')
for n = 1:N
    display(['......speaker ' num2str(playChanList(n))])
    % Total number of pulses
    n_pulse = floor(npt(n)*time);
    % Length of pulse in seconds
    tnp = time/n_pulse;
    % Bandpass filter
    [b,a]=butter(2,[pulseFreq(n)-100 pulseFreq(n)+100]/(sampRate/2));
    % Time vector
    t = (-0.5/npt(n):1/sampRate:0.5/npt(n))';
    % Signal
    h = sin(2*pi*pulseFreq(n)*t).*exp(-(t).^2/2/pulseWidth(n)^2);
    % Remove near zero parts for computational speed
    i1 = find(abs(h)>1e-9,1,'first');
    i2 = find(abs(h)>1e-9,1,'last');
    t = t(i1:i2);
    h = h(i1:i2);

    for m=1:M
        display(['.........microphone ' num2str(recChanList(m))])
        % filter the signal through bandpass filter
        y = recBuffer(:,m);
        yf=filtfilt(b,a,y);

        % convolute with signal
        yc=conv(yf,h);
        ycabs=abs(yc);

        % Find the pulse indices
        minDist = sampRate*tnp*0.5;
        pidx = SFlabFindPulseInd(ycabs,n_pulse,minDist);
        % Calculate the times
        tphat=pidx(1:n_pulse)/sampRate-t(end-1);
        % Sort pulse times
        TPHAT(1:n_pulse,m,n) = sort(tphat);

        % Create plots
        if plotOn
            figure(100),clf
            subplot(3,1,1)
            plot(recBuffer(:,m))
            title(['Signal from microphone ' num2str(recChanList(m))])
            subplot(3,1,2)
            plot(yf), title('Bandpass filtered signal')
            subplot(3,1,3)
            plot(ycabs)
            title(['Filtered signal convoluted with pulse ' num2str(n)])
            hold on
            for k = 1:length(pidx)
                plot(pidx(k),ycabs(pidx(k)),'d','color','r')
            end
            hold off
            display('.........press any key to continue. N.B. the figure will be overwritten!')
            pause
        end
    end
end
display(['...pulse times from ' num2str(N)...
    ' speakers found in ' num2str(M) ' microphones'])
return